package cz.drg.clasificator.writers;

import cz.drg.clasificator.exception.ShutdownException;
import cz.drg.clasificator.setting.ProgramSettings;
import cz.drg.clasificator.setting.Settings;
import cz.drg.clasificator.util.Constants;
import cz.drg.clasificator.util.StringUtil;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import static cz.drg.clasificator.util.OutputHelper.*;

public class CsvWriter extends BaseWriter{

    static{loadDefaults();}
    
    private static String DEFAULT_CHARSET;
    private static String DEFAULT_DELIMITER;
    private static String DEFAULT_OUTPUT_FILEPATH;
    private static File DEFAULT_OUTPUT = new File(DEFAULT_OUTPUT_FILEPATH);
    
    private File target;
    private Charset charset;
    private boolean alreadyWrittenTo = false;
    
    public CsvWriter(File target, String delimiter, String charset) {
        super(delimiter);
        this.target = target;
        this.charset = Charset.forName(charset);
    }
    
    //<editor-fold defaultstate="collapsed" desc="USED IN ARG EVALUATOR FACTORY">
    /**
     * This constructor is used in current implementation of evaluator factory.
     * Based on type and number of parameters the correct constructor is
     * selected from this class. Number of parameters depends on number of user
     * input argument parameters. In case no constructor can be selected the
     * program ends.
     *
     * @param filePath csv input file path
     * @param delimiter delimiter used to read this csv file
     * @param charset charset used to read this csv file
     */
    public CsvWriter(String filePath, String delimiter, String charset) {
        this(new File(filePath), delimiter, charset);
    }
    /**
     * This constructor is used in current implementation of evaluator factory.
     * Based on type and number of parameters the correct constructor is
     * selected from this class. Number of parameters depends on number of user
     * input argument parameters. In case no constructor can be selected the
     * program ends.
     *
     * @param filePath csv input file path
     * @param delimiter delimiter used to read this csv file
     */
    public CsvWriter(String filePath, String delimiter) {
        this(new File(filePath), delimiter, DEFAULT_CHARSET);
    }
    /**
     * This constructor is used in current implementation of evaluator factory.
     * Based on type and number of parameters the correct constructor is
     * selected from this class. Number of parameters depends on number of user
     * input argument parameters. In case no constructor can be selected the
     * program ends.
     *
     * @param filePath csv input file path
     */
    public CsvWriter(String filePath){
        this(new File(filePath), DEFAULT_DELIMITER, DEFAULT_CHARSET);
    }
    /**
     * This constructor is used in current implementation of evaluator factory.
     * Based on type and number of parameters the correct constructor is
     * selected from this class. Number of parameters depends on number of user
     * input argument parameters. In case no constructor can be selected the
     * program ends.
     */
    public CsvWriter() {
        this(DEFAULT_OUTPUT, DEFAULT_DELIMITER, DEFAULT_CHARSET);
    }
    
    //</editor-fold>
    
    /**
     * Init class default values. This expects settings to be loaded before use.
     */
    private static void loadDefaults(){
        ProgramSettings programSettings = Settings.getProgramSettings();
        
        DEFAULT_CHARSET = programSettings.getDefaultCharset();
        DEFAULT_DELIMITER = StringUtil.unescapeJavaString(programSettings.getDefaultDelimiter());
        DEFAULT_OUTPUT_FILEPATH = programSettings.getDefaultOutputPath();
    }

    @Override
    public void writeOutput(List<String> lines) {
        
        if(alreadyWrittenTo){
            lines.remove(0);
        }
        
        StringBuilder sb = new StringBuilder();
        
        lines.stream().forEach(value -> {
            
            sb.append(value);
            sb.append(System.lineSeparator());
            
        });
        
        try {
            Files.write(target.toPath(), sb.toString().getBytes(charset), StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.APPEND);
            alreadyWrittenTo = true;
        } catch (IOException ex) {
            Logger.getLogger(CsvWriter.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    @Override
    public void clear() {
        
        try {
            Files.deleteIfExists(target.toPath());
            alreadyWrittenTo = false;
        } catch (IOException ex) {
            dualLog(ex.getMessage());
            throw new ShutdownException(Constants.ERR_CLEAR_CSV);
        }
    }

    @Override
    public void close() {}

    }
