package cz.drg.clasificator.util;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.databind.AnnotationIntrospector;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.databind.introspect.JacksonAnnotationIntrospector;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import com.fasterxml.jackson.dataformat.xml.ser.ToXmlGenerator;
import com.fasterxml.jackson.module.jaxb.JaxbAnnotationIntrospector;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import javax.xml.parsers.DocumentBuilderFactory;
import org.dmg.pmml.Array;
import org.dmg.pmml.Constant;
import org.dmg.pmml.DataType;
import org.dmg.pmml.Row;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 *
 * @author jiras
 */
public class JacksonUtil {
    
    private static XmlMapper mapper;
    
    public static XmlMapper mapper(){
        if(mapper == null) 
            mapper = createMapper(srcMapper -> AnnotationIntrospector.pair(new JacksonAnnotationIntrospector(), new JaxbAnnotationIntrospector(srcMapper.getTypeFactory())));
        return mapper;
    }
    
    private static XmlMapper createMapper(Function<XmlMapper, AnnotationIntrospector> getIntrospector){
        XmlMapper mapper = XmlMapper.builder()
                .defaultUseWrapper(false)
                .enable(SerializationFeature.INDENT_OUTPUT)
                .enable(ToXmlGenerator.Feature.WRITE_XML_DECLARATION)
                .build();
        
        
        mapper.getFactory().getXMLOutputFactory().setProperty("org.codehaus.stax2.automaticNsPrefix", "");
        
        SimpleModule module = new SimpleModule();
        module.addDeserializer(Array.class, new StdDeserializer<Array>(Array.class) {
            @Override
            public Array deserialize(JsonParser jp, DeserializationContext dc) throws IOException, JsonProcessingException {
                TreeNode readTree = jp.getCodec().readTree(jp);
                
                Array array = new Array(Array.Type.STRING, "  ");
                
                Iterator<String> fieldNames = readTree.fieldNames();
                while (fieldNames.hasNext()) {
                    String field = fieldNames.next();
                    switch(field){
                        case "type": {array.setType(Array.Type.fromValue(((TextNode)readTree.get(field)).asText())); break;}
                        case "" : {array.setValue(((TextNode)readTree.get(field)).asText()); break;}
                        default: {break;}
                    }
                    
                }
                
                return array;
            }
        });
        
        module.addDeserializer(Row.class, new StdDeserializer<Row>(Row.class) {
            @Override
            public Row deserialize(JsonParser jp, DeserializationContext dc) throws IOException, JsonProcessingException {
                TreeNode readTree = jp.getCodec().readTree(jp);
                
                
                ObjectNode contentNode = (ObjectNode) readTree.get("content");
                Iterator<Map.Entry<String, JsonNode>> elements = contentNode.fields();
                List<Object> content = new ArrayList<>();
                
                  try {
                    DocumentBuilderFactory builderFactory = DocumentBuilderFactory.newInstance();
                    builderFactory.setNamespaceAware(false);
                    builderFactory.setValidating(false);
                    Document document = builderFactory.newDocumentBuilder().newDocument();
                    
                    elements.forEachRemaining(entry -> {
                        Element rowContent = document.createElement(entry.getKey());
                        rowContent.setTextContent(entry.getValue().textValue());
                        content.add(rowContent);
                    });
                    

                  } catch (Exception e) {
                    throw new RuntimeException(e);
                  }
                
                return new Row(content);
            }
        });
              
                
        module.addDeserializer(Constant.class, new StdDeserializer<Constant>(Constant.class) {
            @Override
            public Constant deserialize(JsonParser jp, DeserializationContext dc) throws IOException, JsonProcessingException {
                TreeNode readTree = jp.getCodec().readTree(jp);
                
                Constant constant = new Constant();
                
                TreeNode dataTypeNode = readTree.get("dataType");
                if(dataTypeNode != null){
                    constant.setDataType(DataType.fromValue(((TextNode)dataTypeNode).asText()));
                }
                TreeNode valueNode = readTree.get("");
                if (valueNode != null) {
                    constant.setValue(((TextNode)valueNode).asText());
                }
                
                return constant;
            }
        });
        
        mapper.registerModule(module);
        
        mapper.setAnnotationIntrospector(getIntrospector.apply(mapper));
        mapper.configure(MapperFeature.ACCEPT_CASE_INSENSITIVE_PROPERTIES, true);
        mapper.setSerializationInclusion(JsonInclude.Include.NON_EMPTY);
        
        return mapper;
    }
    
}
