package cz.drg.clasificator.util;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 *
 * @author jiras
 */
public class InsertQuery {
    
    public enum DatabaseType{
        
        ORACLE("TO_DATE('%s','yyyymmdd')"), 
        INFORMIX("TO_DATE('%s','%%Y%%m%%d')"), 
        DEFAULT("'%s'");
        
        private String dateFormat;

        private DatabaseType(String dateFormat) {
            this.dateFormat = dateFormat;
        }

        public String getDateFormat() {
            return dateFormat;
        }
        
    }
    
    private static String INSERT = "INSERT INTO %s %s VALUES %s";
    
    private Headers headers;
    private Values values;
    private String targetTable;
    
    private Set<Integer> dateColumns;
    private DatabaseType dbType = DatabaseType.DEFAULT;
    
    InsertQuery(String targetTable) {
        this.targetTable = targetTable;
    }

    void setValues(List<String> row) {
        this.values = new Values(row);
    }
    
    void setHeaders(List<String> headers) {
        this.headers = new Headers(headers);
        setDateColumns(headers);
    }

    private void setDateColumns(List<String> headers){
        dateColumns = new HashSet<>();
        
        for (int i = 0; i < headers.size(); i++) {
            
            String header = headers.get(i);
            
            if(header.equalsIgnoreCase("datum_pri")){
                dateColumns.add(i);
            }
            if(header.equalsIgnoreCase("datum_pro")){
                dateColumns.add(i);
            }
        }
    }
    
    private boolean isNumeric(String str) {
        return str.matches("(0(\\.\\d+)?$)|-?[1-9]\\d*(\\.\\d+)?$");  //match a number with optional '-' and decimal.
    }
    
    public String arrayToValuesString(List<? extends Object> array, String quotation){
        StringBuilder sb = new StringBuilder();
        
        for (int i = 0; i < array.size(); i++) {
            Object obj = array.get(i);
            
            if(obj == null || obj.equals("")){
                sb.append((Object)null);
            }
            else if(dateColumns.contains(i)){
                //special Oracle,Informix date formatting
                sb.append(String.format(dbType.getDateFormat(), obj));
            }
            else if(isNumeric(obj.toString())){
                sb.append(obj.toString());
            }
            else{
                sb.append(quotation);
                sb.append(obj.toString());
                sb.append(quotation);
            }
            
            if(i < array.size() - 1){
                sb.append(",");
            }
        }
        
        return sb.toString();
    }
    
    private String arrayToValues(List<? extends Object> array, String quotation){
        
        StringBuilder sb = new StringBuilder("(");
        
        sb.append(arrayToValuesString(array, quotation));
        
        sb.append(")");
        
        return sb.toString();
    }

    void setDbType(DatabaseType dbType) {
        this.dbType = dbType;
    }

    @Override
    public String toString() {
        return String.format(INSERT, targetTable, headers, values);
    }
    
    private String arrayToString(List<? extends Object> array){
        
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < array.size(); i++) {
            
            Object obj = array.get(i);
            
            sb.append(obj.toString());
            
            if(i < array.size() - 1){
                sb.append(",");
            }
            
        }
        return sb.toString();
    }
    
    private class Headers {

        List<String> headers;

        public Headers(List<String> headers) {
            this.headers = headers;
        }
        
        @Override
        public String toString() {
            
            StringBuilder sb = new StringBuilder("(");
            
            sb.append(arrayToString(headers));
            sb.append(")");
            
            return sb.toString();
        }
    }
    
    private class Values {

        List<String> rowValues;
        
        public Values() {
            rowValues = new ArrayList<>();
        }
        
        public Values(List<String> valuesList) {
            rowValues = valuesList;
        }

        @Override
        public String toString() {
            return arrayToValues(rowValues, "'");
        }
        
    }
    
}
