package cz.drg.clasificator.setting;

import cz.drg.clasificator.setting.program.ACTIONSTYPE;
import cz.drg.clasificator.setting.program.Arguments.Argument;
import cz.drg.clasificator.setting.program.DatabaseDrivers;
import cz.drg.clasificator.setting.program.IOclass;
import cz.drg.clasificator.setting.program.Program;
import java.util.ArrayList;
import java.util.List;

/**
 * Class representing application settings which contains mapping for readers/writers
 * on the datatype thats being read/written.
 * 
 * @author Pavel Jirasek
 */
public class ProgramSettings {

    private final Program base;

    public ProgramSettings(Program base) {
        this.base = base;
    }
    
    /**
     * Gets list of available arguments from loaded program settings.
     * 
     * @return 
     */
    public List<Argument> getArguments(){
        return base.getArguments().getArgument();
    }
    
    /**
     * Gets IO class from loaded program settings based on argument and datatype. 
     * This expects that there cannot be 2 readers/writers for same combination
     * of argument and datatype.
     * <p>
     * Typical argument would be: -|ARGUMENT| |PARAMETER1| |PARAMETER2| ...
     * 
     * @param arg program argument
     * @param type program parameter datatype
     * @return 
     */
    public IOclass getIOclass(String arg, String type){
        List<IOclass> supportedIO = base.getIOclasses().getIOclass();
        
        for (IOclass iOclas : supportedIO) {
            
            if(iOclas.getForArg().equalsIgnoreCase(arg) && iOclas.getForType().equalsIgnoreCase(type)){
                return iOclas;
            }
            
        }
        
        return null;
    }
    
    public String getDefaultPmmlPath(){
        return base.getDefaultPmmlInput();
    }
    public String getDefaultInputPath(){
        return base.getDefaultInput();
    }
    public String getDefaultOutputPath(){
        return base.getDefaultOutput();
    }
    public String getDefaultCharset(){
        return base.getDefaultCharset();
    }
    public String getDefaultDelimiter(){
        return base.getDefaultDelimiter();
    }
    public int getEvaluationBatchSize(){
        return Integer.parseInt(base.getEvaluationBatchSize());
    }
    
    public List<String> getDatabaseDrivers(){
        DatabaseDrivers databaseDrivers = base.getDatabaseDrivers();
        
        if(databaseDrivers == null){
            return new ArrayList<>();
        }
        return base.getDatabaseDrivers().getDriverClass();
    }
    
    /**
     * Gets if the datatype inputed by user is supported by application. Support of the
     * datatype is determined by program settings file.
     * 
     * @param type datatype in question
     * @return 
     */
    public boolean isSupportedType(String type){
        
        for (String supportedVal : base.getIOtypes().getIOtype()) {
            if(supportedVal.equalsIgnoreCase(type)){
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Gets if the argument inputed by user is supported by application. Support of the 
     * argument us determined by program settings file.
     * 
     * @param argument argument in question
     * @return 
     */
    public boolean isSupportedArgument(String argument){
        
        for (Argument supportedVal : base.getArguments().getArgument()) {
            if(supportedVal.getValue().equalsIgnoreCase(argument)){
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Gets if the action inputed by user is supported by application. Support of the 
     * action us determined by program settings file.
     * 
     * @param action
     * @return 
     */
    public boolean isSupportedAction(String action){
        
        for (ACTIONSTYPE supportedVal : base.getActions().getAction()) {
            if(supportedVal.value().equalsIgnoreCase(action)){
                return true;
            }
        }
        
        return false;
    }
    
}
