package cz.drg.clasificator.readers;

import cz.drg.clasificator.setting.ProgramSettings;
import cz.drg.clasificator.setting.Settings;
import cz.drg.clasificator.util.Constants;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.UnsupportedCharsetException;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import static cz.drg.clasificator.util.OutputHelper.*;
import java.util.zip.ZipFile;

/**
 * Implementation of reader for "csv" datatype.
 *
 * @author Pavel Jirasek
 */
public class CsvReader extends BaseReader {

    static {
        loadDefaults();
    }

    private static String DEFAULT_CHARSET;
    private static String DEFAULT_DELIMITER;
    private static String DEFAULT_INPUT_FILEPATH;
    private static String DEFAULT_PMML_INPUT_FILEPATH;

    private static File DEFAULT_INPUT;
    private static ZipFile DEFAULT_PMML_INPUT;

    private static Charset CHARSET;

    private final int batchSize;
    private boolean hasNextBatch = true;
    private boolean isFirstBatch = true;
    private BufferedReader reader;

    public CsvReader(File input, ZipFile pmmlInput, String delimiter, String charset) {
        super(pmmlInput, delimiter);
        batchSize = Settings.getProgramSettings().getEvaluationBatchSize();
        
        
        try {
            CHARSET = Charset.forName(charset);
        } catch (UnsupportedCharsetException ex) {
            dualLog(String.format(Constants.ERR_UNSUPPORTED_CHARSET_NAME, charset, DEFAULT_CHARSET));
            CHARSET = Charset.forName(DEFAULT_CHARSET);
        }

        try {
            reader = Files.newBufferedReader(input.toPath(), CHARSET);
        } catch (IOException ex) {
            dualLog("Error occured while opening the file "+input.getPath()+".");
            dualLog(ex.getMessage());
            System.exit(500);
        }
        
        init();
    }

    public CsvReader(File input, String delimiter, String charset) {
        this(input, DEFAULT_PMML_INPUT, delimiter, charset);
    }

    public CsvReader(File source) {
        this(source, DEFAULT_DELIMITER, DEFAULT_CHARSET);
    }

    public CsvReader(File input, ZipFile pmmlInput) {
        this(input, pmmlInput, DEFAULT_DELIMITER, DEFAULT_CHARSET);
    }

    //<editor-fold defaultstate="collapsed" desc="USED IN ARG EVALUATOR FACTORY">
    /**
     * This constructor is used in current implementation of evaluator factory.
     * Based on type and number of parameters the correct constructor is
     * selected from this class. Number of parameters depends on number of user
     * input argument parameters. In case no constructor can be selected the
     * program ends.
     *
     * @param input csv input file path
     * @param delimiter delimiter used to read this csv file
     * @param charset charset used to read this csv file
     */
    public CsvReader(String input, String delimiter, String charset) {
        this(new File(input), DEFAULT_PMML_INPUT, delimiter, charset);
    }

    /**
     * This constructor is used in current implementation of evaluator factory.
     * Based on type and number of parameters the correct constructor is
     * selected from this class. Number of parameters depends on number of user
     * input argument parameters. In case no constructor can be selected the
     * program ends.
     *
     * @param input csv input file path
     * @param delimiter delimiter used to read this csv file
     */
    public CsvReader(String input, String delimiter) {
        this(new File(input), DEFAULT_PMML_INPUT, delimiter, DEFAULT_CHARSET);
    }

    /**
     * This constructor is used in current implementation of evaluator factory.
     * Based on type and number of parameters the correct constructor is
     * selected from this class. Number of parameters depends on number of user
     * input argument parameters. In case no constructor can be selected the
     * program ends. Since the delimiter and charset are ommited default values
     * takes place.
     *
     * @param input csv input file path
     */
    public CsvReader(String input) {
        this(new File(input), DEFAULT_PMML_INPUT, DEFAULT_DELIMITER, DEFAULT_CHARSET);
    }

    /**
     * This constructor is used in current implementation of evaluator factory.
     * Based on type and number of parameters the correct constructor is
     * selected from this class. Number of parameters depends on number of user
     * input argument parameters. In case no constructor can be selected the
     * program ends. Since all parameters are ommited default values takes
     * place.
     */
    public CsvReader() {
        this(DEFAULT_INPUT, DEFAULT_DELIMITER, DEFAULT_CHARSET);
    }
    //</editor-fold>

    /**
     * Init class default values. This expects settings to be loaded before use.
     */
    private static void loadDefaults() {
        ProgramSettings programSettings = Settings.getProgramSettings();

        DEFAULT_INPUT_FILEPATH = programSettings.getDefaultInputPath();
        DEFAULT_PMML_INPUT_FILEPATH = programSettings.getDefaultPmmlPath();
        DEFAULT_CHARSET = programSettings.getDefaultCharset();
        DEFAULT_DELIMITER = programSettings.getDefaultDelimiter();
        
        DEFAULT_INPUT = new File(DEFAULT_INPUT_FILEPATH);
        try {
            DEFAULT_PMML_INPUT = new ZipFile(DEFAULT_PMML_INPUT_FILEPATH);
        } catch (IOException ex) {
            dualLog("Error loading default PMML zip file: '"+DEFAULT_PMML_INPUT_FILEPATH+"'");
            System.exit(-1);
        }
    }

    private long totalEntries = -1;
    private String readAhead = null;
    
    @Override
    protected List<String> getData() {
        
        List<String> result = new ArrayList<>();
        
        int size = isFirstBatch ? batchSize+1 : batchSize;
        
        for (int i = 0; i < size; i++) {
            String line = null;
            
            //check one line ahead in case number of lines % batchsize = 0
            //to end the reading in case next batch would contain no items
            if(readAhead != null){
                result.add(readAhead);
                readAhead = null;
                continue;
            }
            
            try {
                line = reader.readLine();
            } 
            catch (IOException ex) {
                dualLog("Error occured while reading the file.");
                dualLog(ex.getMessage());
                return result;
            }
            
            if (line != null) {
                result.add(line);
            } 
            else {
                break;
            }
        }
        
        totalEntries += result.size();
        
        isFirstBatch = false;
        
        try {
            readAhead = reader.readLine();
            hasNextBatch = readAhead != null;
        } 
        catch (IOException ex) {
            Logger.getLogger(CsvReader.class.getName()).log(Level.SEVERE, null, ex);
        }
        
        return result;
    }

    @Override
    public boolean hasNextEntryBatch() {
        return hasNextBatch;
    }

    @Override
    public void close() {
        try {
            reader.close();
        } catch (IOException ex) {
            Logger.getLogger(CsvReader.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    
}
