package cz.drg.clasificator.args.argevaluation;

import cz.drg.clasificator.args.ProgramExecutor;
import cz.drg.clasificator.readers.InputReader;
import cz.drg.clasificator.setting.Settings;
import cz.drg.clasificator.setting.program.ACTIONSTYPE;
import cz.drg.clasificator.setting.program.Arguments.Argument;
import cz.drg.clasificator.setting.program.IOclass;
import cz.drg.clasificator.util.Constants;
import cz.drg.clasificator.writers.OutputWriter;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Default implementation of evaluator used for all arguments in the application.
 * 
 * @author Pavel Jirasek
 */
public class ArgEvaluator implements Evaluator{

    public static final String HELP_ARG = "help";
    public static final String CONFIG_FILE_ARG = "configFile";
    public static final String DATABASE_ARG = "database";
    
    protected String argument;
    /**
     * Parameters are always in format <type> (<filepath> <delimiter> <charset>) .
     * Only parameter <type> is required and can be used as single parameter. 
     * Other parameters need to be, if used, always in same order as specified, meaning
     * that parameter <delimiter> cannot be used without parameter <filepath> before it.
     * Same applies to <charset> parameter.
     */
    private List<String> parameters;
    
    public ArgEvaluator(String argument) {
        this.argument = argument;
        parameters = new ArrayList<>();
    }

    private void registerDatabaseDrivers(){
        
        try {
            
            List<String> databaseDrivers = Settings.getProgramSettings().getDatabaseDrivers();
            
            for (String databaseDriver : databaseDrivers) {
                //register not included database drivers
                Class.forName(databaseDriver);
            }
            
        } catch (ClassNotFoundException ex) {
            System.err.println("Failed to load database driver.");
            System.err.println(ex.getMessage());
            System.exit(500);
        }
    }
    
    @Override
    public void evaluate(ProgramExecutor visitor) {
        
        if(argument.equalsIgnoreCase(HELP_ARG)){
            visitor.stopExecution(getHelpMsg());
            return;
        }
        else if(argument.equalsIgnoreCase(CONFIG_FILE_ARG)){
            Settings.loadSettings(parameters.get(0));
            return;
        }
        
        if(parameters.isEmpty()){
            visitor.stopExecution(Constants.ERR_ARG_PARAMETER_MISSING);
            return;
        }

        //there is expected that the first parameter of argument is datatype
        String datatype = parameters.get(0);

        IOclass iOclass = Settings.getProgramSettings().getIOclass(argument, datatype);
        
        if(iOclass == null){
            visitor.stopExecution(String.format(Constants.ERR_ARG_TYPE_NOT_SUPPORTED, argument, datatype));
            return;
        }
        
        //if its database input/output, database drivers needs to be loaded before connecting to database
        if(datatype.equalsIgnoreCase(DATABASE_ARG)){
            registerDatabaseDrivers();
        }
        
        
        String classname = iOclass.getClassname();
        Class<?> forName;
        try {
            forName = Class.forName(classname);
        } catch (ClassNotFoundException ex) {
            visitor.stopExecution(Constants.ERR_IO_CLASS_NOT_FOUND);
            return;
        }
        
        Class[] parameterTypes = new Class[parameters.size() - 1];
        for (int i = 0; i < parameterTypes.length; i++) {
            parameterTypes[i] = String.class;
        }
        
        List<String> subList;
        if(parameters.size() - 1 > 0){
            subList = parameters.subList(1, parameters.size());
        }
        else{
            subList = new ArrayList<>();
        }
        
        Object object;
        try{
            //IMPORTANT: ALL READERS/WRITES HAS TO IMPLEMENT CONSTRUCTOR FOR VARIABLE AMMOUNT OF STRINGS FOR THIS TO WORK
            object = forName.getConstructor(parameterTypes).newInstance(subList.toArray());
        }
        catch(NoSuchMethodException|InstantiationException|IllegalAccessException|InvocationTargetException ex){
            visitor.stopExecution(Constants.ERR_WRONG_NUMBER_OF_PARAMS);
            return;
        }
        
        if(object instanceof ConfigurableIOClass){
            ((ConfigurableIOClass)object).setIoClass(iOclass);
        }
        
        ACTIONSTYPE forAction = iOclass.getForAction();
        
        setActionForVisitor(visitor, forAction, object);
        
    }
    
    private void setActionForVisitor(ProgramExecutor visitor, ACTIONSTYPE forAction, Object object){
        
        switch(forAction){
            case READ:
                visitor.setReader((InputReader) object);
                break;
            case WRITE: 
                visitor.setWriter((OutputWriter) object);
                break;
                
            default: visitor.stopExecution(String.format(Constants.ERR_ACTION_NOT_SUPPORTED, forAction));
        }
        
    }
    
    private String getHelpMsg(){
        
        String supportedArgs = "Supported program arguments:\n";
        
        for (Argument argument : Settings.getProgramSettings().getArguments()) {
            supportedArgs += String.format("-%-10s%s\n", argument.getValue(), argument.getDescription());
        }
        
        
        return supportedArgs;
    }
    
    
    @Override
    public void addParameter(String param) {
        parameters.add(param);
    }
    
    public Evaluator clearParameters(){
        parameters.clear();
        return this;
    }
    
    @Override
    public String toString() {
        
        String result = "-"+argument;
        
        for (String parameter : parameters) {
            result += " "+parameter;
        }
        
        return result;
    }
}
